/*
    This file is a part of the program gateToFermat.
    Copyright (C) Mikhail Tentyukov <tentukov@physik.uni-bielefeld.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License version 2 as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
*/

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include "gateToFermat.h"

/*
This is the wrapper to the program FERMAT
http://www.bway.net/~lewis

There are the following environment variables recognized by the program:

GTF_THEPATH - full path to the Fermat executable.

GTF_FASTPOLYGCD  - if 0, then the  fastest shortcut polygcd is off; 
otherwise, it is on (default) (it is buggy in fermat-3.6.4!)
*/

#define DELTA_OUT 128
#define FROMFERMATBUFSIZE 1024

#define WAITPID(pid)	waitid(pid,P_PID,NULL,0)
// #define WAITPID(pid)	waitpid(pid,NULL,0)

static char *g_thepath=THEPATH;


/*The output buffer (baseout) and related control pointers.
  stopout points to the end of allocated space, fullout points to 
  the end of space filled by an actual content:*/
static char *g_baseout=NULL,
     *g_fullout=NULL,
     *g_stopout=NULL;

/*The inline function places one char to the output buffer with possible 
  expansion of the buffer:*/
static void addtoout(char ch)
{
   if (g_fullout >= g_stopout){
      int l=g_stopout-g_baseout+DELTA_OUT;
      char *ptr=realloc(g_baseout,l);
      if(ptr==NULL)
         exit(2);
      g_fullout=ptr+(g_fullout-g_baseout);
      g_stopout=ptr+l;
      g_baseout=ptr;
   }/*if (g_fullout >= g_stopout)*/
   *g_fullout++ = ch;
}/*addtoout*/

static FILE *g_to=NULL, *g_from=NULL;
static pid_t  g_childpid;
static char g_fbuf[FROMFERMATBUFSIZE];


/*
M.Tentyukov, 30.07.2008 -- the problem with Fermat running from the
program communicating via MathLink: Fermat ignores SIGPIPE so it
continues to run after the main program dies.

atexit() does not help: the main program may be terminated abnormally.

So we fork out the guard process which is listening to the main
program through the pipe.  The guard ignores SIGPIPE, so after it
returns form the reading the pipe this means that the main program 
is dead. The guard kill Fermat and exit.

The main program sets SIGCHLD handler so that if the guard is killed,
it kills Fermat and exit.

*/

/*INTSIGHANDLER : some systems require a signal handler to return an integer,
  so define the macro INTSIGHANDLER if compiler fails:*/
#ifdef INTSIGHANDLER
static int sigCHL(int i)
#else
static void sigCHL(int i)
#endif
{
   /*This handler triggered when the child dies.
     The program has only two children, Fermat and the guard.
     If one if them dies, this Fermat has to be killed.
    */
   kill(g_childpid,SIGKILL);
   exit(0);
#ifdef INTSIGHANDLER
   return 0;
#endif
}/*sigCHL*/

/*The function startGuard returns 0 on success, or -1.*/

static int startGuard(void)
{
int fdin[2];
pid_t childpid;
     pipe(fdin);
     if((childpid = fork()) == -1){
        perror("fork");
        return(-1);
     }/*if((childpid = fork()) == -1)*/

     if(childpid == 0){
        /* Child process closes up input side of pipe */
        close(fdin[1]);
        /*Now we may read from fdin[0]*/
        /* if compiler fails here, try to define INTSIGHANDLER 
           in the beginning of this file:*/
        signal(SIGPIPE,SIG_IGN);/*Survive on the pipe closing*/
        /*Use childpid as a buffer -- we need not it anymore:*/
        read(fdin[0], &childpid, sizeof(pid_t));
        /*The father newer writes to the fdin[1] so if we here 
          then the father is dead.*/
        kill(g_childpid,SIGKILL);
        exit(0);
     }/*if(childpid == 0)*/

     /* Parent process closes up output side of pipe */
     close(fdin[0]);
     signal(SIGCHLD, sigCHL);
     return(0);
}/*startGuard*/


/*Starts Fermat and swallows its stdin/stdout:*/
static pid_t  openprogram(FILE **to, FILE **from)
{
int fdin[2], fdout[2];
pid_t childpid;
     pipe(fdin);
     pipe(fdout);
     if((childpid = fork()) == -1){
        perror("fork");
        return(0);
     }/*if((childpid = fork()) == -1)*/

     if(childpid == 0){
        /* Child process closes up input side of pipe */
        close(fdin[1]);
        close(fdout[0]);

        /*reopen stdin:*/
        close(0);
        dup(fdin[0]);
        /*reopen stdout:*/
        close(1);
        dup(fdout[1]);
        /*stderr > /dev/null*/
        close(2);/*close stderr*/
        open("/dev/null",O_WRONLY); /* reopen stderr*/
        /*argc[0] = must be the same as g_thepath: 
          Fermat uses it to find the full path:*/
        execlp(g_thepath, g_thepath, NULL);
        /*perror? -nonsense! stdout > /dev/null!*/
        /*perror("execlp");*/

        /*Error?*/
        /*30.07.08, M.Tentyukov: return here is a nonsense! Exit!*/
        /*return(0);*/
        exit(0);
     }else{ /* Parent process closes up output side of pipe */
        close(fdin[0]);
        close(fdout[1]);
        if (  (*to=fdopen(fdin[1],"w"))==NULL){
           kill(childpid,SIGKILL);
           return(0);
        }/*if (  (*to=fdopen(fdin[1],"w"))==NULL)*/
        if (  (*from=fdopen(fdout[0],"r"))==NULL){
           fclose(*to);
           kill(childpid,SIGKILL);
           return(0);
	     }/*if (  (*from=fdopen(fdout[0],"r"))==NULL)*/
     }
     return childpid;
}/*openprogram*/

/*reads the stream 'from' up to the line 'terminator' (only 'thesize' first 
  characters are compared):*/
static void readup(FILE *from, char *terminator, int thesize)
{
char *c;
   for(;;){
      do{
         for(c=fgets(g_fbuf,FROMFERMATBUFSIZE,g_from); *c<=' ' ; c++)
            if(*c == '\0')break;
      }while(*c<=' ');
      if(strncmp(terminator,c,thesize)==0)
         return;
   }
}/*readup*/

/*Starts Fermat and makes some initializations:*/
static pid_t initFermat(FILE **to, FILE **from, char *pvars)
{
pid_t   childpid;
char *ch;
   childpid=openprogram(to, from);
   if(  childpid==0)
      return -1;
   /*Fermat is running*/

   /*Switch off floating point representation:*/
   fputs("&d\n0\n",*to);

   /*Set prompt as '\n':*/
   fputs("&(M=' ')\n",*to);
   fflush(*to);
   readup(*from,"> Prompt",8);
   readup(*from,"Elapsed",7);
   fgets(g_fbuf,FROMFERMATBUFSIZE,*from);

   /*Switch off timing:*/
   fputs("&(t=0)\n",*to);
   fflush(*to);
   readup(*from,"0",1);
   fgets(g_fbuf,FROMFERMATBUFSIZE,*from);

   /*Switch on "ugly" printing: no spaces in long integers;
     do not omit '*' for multiplication:*/
   fputs("&U\n",*to);
   fflush(*to);
   readup(*from,"0",1);
   fgets(g_fbuf,FROMFERMATBUFSIZE,*from);
    
   /*NEW*/
   /*Switch off suppression of output to terminal of long polys.:*/
   fputs("&(_s=0)\n",*to);
   fflush(*to);
   readup(*from,"0",1);
   fgets(g_fbuf,FROMFERMATBUFSIZE,*from);
	

   /*Set polymomial variables:*/
   /*pvars looks like "a\nb\nc\n\n":*/
   while( *pvars > '\n'){
      /*Copy the variable up to '\n' into g_fbuf:*/
	/* Modification by A.Smirnov: for compatibility with Mathematica*/
      for(*(ch=g_fbuf)='\0';((*ch=*pvars)>'\n' && (*ch=*pvars)!=',');ch++,pvars++)
         if(ch-g_fbuf >= FROMFERMATBUFSIZE)
            return -2;
      *ch='\0';
      if(*pvars!='\0')pvars++;
      /*Set g_fbuf as a polymomial variable:*/
      fprintf(*to,"&(J=%s)\n",g_fbuf);
      fflush(*to);
      readup(*from,"0",1);
      fgets(g_fbuf,FROMFERMATBUFSIZE,*from);
   }/*while( *pvars > '\n')*/

   ch=getenv("GTF_FASTPOLYGCD");
   if(ch!=NULL){
      if( (*ch=='0')&&(ch[1]=='\0') ){
         /*Switch off fastest shortcut polygcd method (it is buggy in 
           fermat-3.6.4!):*/
         fputs("&(_t=0)\n",*to);
         fflush(*to);
         readup(*from,"0",1);
      }else{
         /*Switch on fastest shortcut polygcd method (default, BTW):*/
         fputs("&(_t=1)\n",*to);
         fflush(*to);
         readup(*from,"1",1);
         fgets(g_fbuf,FROMFERMATBUFSIZE,*from);
      }               
   }/*if(ch!=NULL)*/

   return childpid;
}/*initFermat*/

/*Public function:*/
/* Make some initializations and start Fermat. pvars is a 
   list of polynomial variables separated by '\n', like: "a\nb\nc"*/
int iniCalc(char* path,char *pvars)
{
char *ch;

   if(  (   (ch=getenv("GTF_THEPATH"))!=NULL  )&&
         (*ch != '\0')
     )
      g_thepath=ch;
   if( *path != '\0')
      g_thepath=path;

   if( (g_childpid=initFermat(&g_to, &g_from, pvars))<0 ) /*Error?*/
      return (int)g_childpid;

   /*Allocate output buffer:*/
   if( (g_baseout=malloc(DELTA_OUT))!=NULL ){   
      g_fullout=g_baseout;
      g_stopout=g_baseout+DELTA_OUT;
   }else{/*Memory allocation error, kill started program and exit:*/
      kill(g_childpid,SIGKILL);
  //    waitpid(g_childpid,NULL,0);
      WAITPID(g_childpid);
      return -1;
   }
   if(startGuard()){/*Something is wrong, die*/
      kill(g_childpid,SIGKILL);
    //  waitpid(g_childpid,NULL,0);
      WAITPID(g_childpid);
      return -1;
   }
   return 0;
}/*initCalc*/

/*Add new variables to fermat*/
int addVars(char *pvars)
{
char *ch;
   while( *pvars > '\n'){
      /*Copy the variable up to '\n' into g_fbuf:*/
	/* Modification by A.Smirnov: for compatibility with Mathematica*/
      for(*(ch=g_fbuf)='\0';((*ch=*pvars)>'\n' && (*ch=*pvars)!=',');ch++,pvars++)
         if(ch-g_fbuf >= FROMFERMATBUFSIZE)
            return -2;
      *ch='\0';
      if(*pvars!='\0')pvars++;
      /*Set g_fbuf as a polymomial variable:*/
      fprintf(g_to,"&(J=%s)\n",g_fbuf);
      fflush(g_to);
      readup(g_from,"0",1);
      fgets(g_fbuf,FROMFERMATBUFSIZE,g_from);
   }/*while( *pvars > '\n')*/
return 0;
} /* addVars */





/*Public function:*/
char *calc(char *buf)
{
char *c;
   /*Feed the buffer to Fermat:*/
   for(c=buf; *c!='\0';c++)               
      if (*c!=' ') putc(*c,g_to);
   putc('\n',g_to);/*stroke the line*/
   fflush(g_to);/*Now Fermat starts to work*/

   /*Read the Fermat answer (up to "\n\n") and collect everything 
      into the output buffer:*/
   /*Ignore leading empty lines:*/
   do{
      for(c=fgets(g_fbuf,FROMFERMATBUFSIZE,g_from); *c<=' ' ; c++)
          if(*c == '\0')break;
   }while(*c<=' ');

   /*initialize the output buffer:*/
   g_fullout=g_baseout;

   do{
      /*ignore '`' and spaces:*/
      for(;*c!='\n';c++)switch(*c){
         case ' ':case '`':continue;
         default:addtoout(*c);
      }/*for(;*c!='\n';c++)switch(*c)*/
   }while( *(c=fgets(g_fbuf,FROMFERMATBUFSIZE,g_from))!='\n' );
   /*the empty line is the Fermat prompt*/

   addtoout('\0');/*Complete the line*/
   return(g_baseout);
}/*calc*/

/*Public function:*/
/*mustCleanup == 0 -- no allocated memory free:*/
void closeCalc(int mustCleanup)
{
   /*Stop Fermat:*/
   fputs("&q\n",g_to);
   fflush(g_to);
   fclose(g_to);
   fclose(g_from);
   kill(g_childpid,SIGKILL);
   WAITPID(g_childpid);
   if(mustCleanup){
      free(g_baseout);
      g_baseout=NULL,
      g_fullout=NULL,
      g_stopout=NULL;
   }
}/*closeCalc*/
